# 3 PHP generator.py
import sys
import time
import pickle
import subprocess
import multiprocessing
import os
import gc
from pathlib import Path

# Iestatām ceļus, lai atrastu mūsu moduļus
SCRIPT_DIR = Path(__file__).parent.resolve()
KODS_DIR = SCRIPT_DIR / 'kods'
sys.path.append(str(KODS_DIR))

# Importējam mūsu jaunos moduļus no kods/py
from py import paths, file_ops, renderer, sitemap

def run_data_prep():
    """Palaiž datu sagatavošanu."""
    prep_script = KODS_DIR / "py" / "1_prepare_data.py"
    if not prep_script.exists():
        print("Kļūda: Nav atrasts sagatavošanas skripts.")
        return False
    
    print("\n=== 1. POSMS: DATU APSTRĀDE ===")
    try:
        subprocess.run([sys.executable, str(prep_script)], check=True)
        return True
    except:
        return False

def main():
    # 1. Datu sagatavošana
    if not run_data_prep(): return
    
    print("\n=== 2. POSMS: LAPU ĢENERĒŠANA ===")
    start = time.time()
    
    # 2. Resursu sinhronizācija
    file_ops.sync_resources()
    
    # 3. Ielādējam reģistru (tikai sarakstam)
    print("   -> Ielasa sarakstu...")
    with open(paths.CACHE_FILE, 'rb') as f:
        reg_df = pickle.load(f)['dataframes']['register']
        gc.collect()
        
    valid_regs = {r for r in set(reg_df['regcode'].dropna()) if str(r).isdigit() and len(str(r)) == 11}
    
    # 4. Lietotāja ievade
    val = input(f"   -> Pieejami {len(valid_regs)} uzņēmumi. 'all' vai reģ. nr: ").strip().lower()
    to_process = sorted(list(valid_regs)) if val == 'all' else ([val] if val in valid_regs else [])
    
    if not to_process:
        print("   -> Nekas nav izvēlēts.")
        return

    # 5. Ģenerēšana (Multiprocessing)
    print(f"   -> Sāk ģenerēt {len(to_process)} lapas...")
    
    # Mac optimizācija: max procesi - 1
    cpu_count = max(1, os.cpu_count() - 1)
    
    with multiprocessing.Pool(processes=cpu_count, initializer=renderer.init_worker, initargs=(paths.CACHE_FILE,)) as pool:
        cnt = 0
        for _ in pool.imap_unordered(renderer.process_company, to_process):
            cnt += 1
            if cnt % 100 == 0: print(f"      Progres: {cnt}/{len(to_process)}...", end="\r")

    # 6. Papildu lapas un SEO
    renderer.render_special_pages()
    
    # Sitemap (tikai aktīvajiem)
    if len(to_process) > 1:
        active = set(reg_df[((~reg_df['closed'].isin(['L','R'])) & (reg_df['terminated'].isna()))]['regcode'])
        sitemap_list = [r for r in to_process if r in active]
        if sitemap_list: sitemap.generate(sitemap_list)

    # 7. JS
    file_ops.download_js_libs()
    
    print(f"\n✅ PABEIGTS! Laiks: {time.time() - start:.1f}s")

if __name__ == "__main__":
    multiprocessing.freeze_support()
    main()